/*
**********************************************************************************************************************
THE WORK (AS DEFINED BELOW) IS PROVIDED UNDER THE TERMS OF THIS CREATIVE COMMONS PUBLIC LICENSE ("CCPL" OR "LICENSE"). THE WORK IS PROTECTED BY COPYRIGHT AND/OR OTHER APPLICABLE LAW. ANY USE OF THE WORK OTHER THAN AS AUTHORIZED UNDER THIS LICENSE OR COPYRIGHT LAW IS PROHIBITED.

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
To view a copy of this license, visit http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to Creative Commons, 543 Howard Street, 5th Floor, San Francisco, California, 94105, USA.
**********************************************************************************************************************
*/

/**
*>*************************************************************************<
* Name : Simplejax Access Library
* Author : Ed Cradock
* Modified : 10 October 2006
* Version : v1.1
* 
* Method : __construct
*
* @param string uri - URI associated to page for data retrieval
* @param object callback - Callback function for successful data retrieval
* @param +object errorHandler - Callback function for error handler
*
* @return void
*
* Description:
* Simplejax library for clean page requests. Supports both POST and GET.
*>**************************************************************************<
*/
function simplejax(uri, callback, errorHandler)
{
   /* Default error handler, can be overloaded */
   if(typeof(errorHandler) == 'function')
   this.error = errorHandler;
   else this.error = function(context){alert(context);}
   
   this.uri = uri;         // Uri to retrieve
   this.methodType = 'GET';   // Method type
   this.postArgs;             // Post arguments
   this.returnXML = false;    // Return XML flag
   this.json = false;
   
   this.callback = callback;        // Callback function
   this.errorHandler = this.error;  // Error handler callback
   
   this.source = false;       // Internal buffer for source
   this.http = false;         // HTTP request object

   var _this = this;          // Registered for methods that lose scope

   /**
   *>***************************************************************<
   * Method : construct
   * Access : PRIVATE
   *
   * @param void
   *
   * @return object ^ bool
   *
   * Description:
   * Object creation method, privately accessed, handles both ActiveX
   * and Netscape's proprietary object
   *>***************************************************************<
   */
   this.construct = function()
   {
      var reqObj = false;
      try{reqObj = new ActiveXObject("Microsoft.XMLHTTP");}
      catch(e){reqObj = new XMLHttpRequest()}

      if(reqObj)
      {
         reqObj.open(this.methodType, this.uri, true);

         if(this.methodType == 'POST')
         {
            reqObj.setRequestHeader('Content-type', 'application/x-www-form-urlencoded');
            var post = true;
         }

         reqObj.send((post == true && postArgs != null) ? this.postArgs : null);

         post = false; this.postArgs = null; this.methodType = 'GET';

         return reqObj;
      }
      else
         this.errorHandler.call(this, 'Failure to create HTTP Request Object, please use a supported browser.');
   }

   /**
   *>***************************************************************<
   * Method : set
   * Access : PUBLIC 
   *
   * @param enum option - set option
   * @param mixed value - Value for option
   * @param +string argument - POST argument string
   *
   * @return void
   *
   * Description:
   * Set options, values are saved in reserved buffer variables and
   * are accessed when invoke method is invoked.
   * 
   * Extra:
   * Available options:
   * string URL - Assign a new url
   * object CALLBACK - Assign a new callback function
   * enum(GET,POST) METHOD - second parameter is the post arguments
   * bool XML - Return content type as XML
   * bool JSON - Parse as JSON and return object
   *>***************************************************************<
   */
   this.set = function(option, value, argument)
   {
      option = option.toLowerCase(); 
      switch(option)
      {
         case 'url':
         this.uri = value;
         break;

         case 'callback':
         this.callback = value;
         break;

         case 'method':
         value = value.toUpperCase();
         this.methodType = (value != 'POST') ? 'GET' : value;
         this.postArgs = argument;
         break;

         case 'xml':
         this.returnXML = (value != true) ? false : value;
         break;

         case 'json':
         this.json = (value != true) ? false : value;
         break;

         default:
         this.errorHandler.call(this, 'Invalid option ' + option);
      }
   }

   /**
   *>***************************************************************<
   * Method : invoke
   * Access : PUBLIC 
   *
   * @param void 
   *
   * @return void ^bool
   *
   * Description:
   * Invoke object creation and establish callback.
   *>***************************************************************<
   */
   this.invoke = function()
   {
      _this.http = _this.construct();

      if(typeof(_this.http) != undefined)
         _this.http.onreadystatechange = _this.getsource;
      else
         return false;
   }

   /**
   *>****************************************************************<
   * Method : getsource
   * Access : PRIVATE 
   *
   * @param void 
   *
   * @return void
   *
   * Description:
   * Retrieves source of URI and stores in associated reserved buffer
   * variable and invokes callback function.
   *>****************************************************************<
   */
   this.getsource = function()
   {
      if(_this.http.readyState == 4 && _this.http.status == 200)
      {
         if(_this.returnXML === true)
            _this.source = _this.http.responseXML;
         else if(_this.json == true)
            _this.source = eval(_this.http.responseText);
         else
            _this.source = _this.http.responseText;

         _this.http = false;
         _this.callback.call(_this, _this.source);
      }
   }
}
