package com.example.statemachine.exception.handling;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatNoException;
import static org.assertj.core.api.AssertionsForClassTypes.assertThatThrownBy;
import static org.assertj.core.api.BDDAssertions.as;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.atLeast;
import static org.mockito.Mockito.atMost;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;

import java.util.ConcurrentModificationException;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import org.assertj.core.api.InstanceOfAssertFactories;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.mockito.Mockito;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.context.annotation.Bean;
import org.springframework.lang.Nullable;
import org.springframework.statemachine.StateContext;
import org.springframework.statemachine.StateMachine;
import org.springframework.statemachine.StateMachineContext;
import org.springframework.statemachine.action.Action;
import org.springframework.statemachine.action.Actions;
import org.springframework.statemachine.config.EnableStateMachineFactory;
import org.springframework.statemachine.config.StateMachineConfigurerAdapter;
import org.springframework.statemachine.config.StateMachineFactory;
import org.springframework.statemachine.config.builders.StateMachineStateConfigurer;
import org.springframework.statemachine.config.builders.StateMachineTransitionConfigurer;
import org.springframework.statemachine.guard.Guard;
import org.springframework.statemachine.state.State;
import org.springframework.statemachine.support.DefaultStateMachineContext;
import org.springframework.test.context.TestConstructor;
import org.springframework.test.context.junit.jupiter.SpringJUnitConfig;

import lombok.AccessLevel;
import lombok.AllArgsConstructor;

@ExtendWith(MockitoExtension.class)
@SpringJUnitConfig(classes = {
		StateMachineExceptionHandlingTest.TestStateMachineConfigurer.class,
		StateMachineExceptionHandlingTest.BeanConfig.class })
@TestConstructor(autowireMode = TestConstructor.AutowireMode.ALL)
class StateMachineExceptionHandlingTest {

	private static final String EXCEPTION_TEST_LABEL = Exception.class.getSimpleName();

	private static final String ERROR_TEST_LABEL = Error.class.getSimpleName();

	private static final RuntimeException ACTION_EXCEPTION = new IllegalArgumentException("Action-Exception");

	private static final Error ACTION_ERROR = new StackOverflowError("Action-Error");

	private static final RuntimeException GUARD_EXCEPTION = new IllegalStateException("Guard-Exception");

	private static final Error GUARD_ERROR = new IllegalAccessError("Guard-Error");

	StateMachineExceptionHandlingTest(
			final StateMachineFactory<TestState, TestEvent> stateMachineFactory,
			final Action<TestState, TestEvent> actionMock,
			final Guard<TestState, TestEvent> guardMock) {
		stateMachine = stateMachineFactory.getStateMachine();
		this.actionMock = actionMock;
		this.guardMock = guardMock;
	}

	private final StateMachine<TestState, TestEvent> stateMachine;

	private final Action<TestState, TestEvent> actionMock;

	private final Guard<TestState, TestEvent> guardMock;

	@BeforeEach
	void resetMockBeans() {
		Mockito.reset(actionMock, guardMock);
	}

	/**
	 * When statemachine is started an exception occurs in its initial action.
	 */
	@Test
	// safe Cast, read-only access
	// deprecation should be removed (see SSM issue 1092)
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testExceptionInInitialAction() {

		// initial Action
		doThrow(ACTION_EXCEPTION).when(actionMock).execute(any(StateContext.class));
		// OK -> container mechanism not needed to expose exception
		assertThatThrownBy(stateMachine::start).isEqualTo(ACTION_EXCEPTION);
		// OK -> transition execution interrupted immediately
		assertThatStateMachineStateIsNull();
		// NOK -> we cannot change null state by stop + reset + start
		assertThatStateMachineResetDoesNotChangeState();
		assertThatExtendedStateHoldsExceptionsInOrder(ACTION_EXCEPTION);
	}

	/**
	 * When statemachine is started an error occurs in its initial action.
	 */
	@Test
	// safe Cast, read-only access
	// deprecation should be removed (see SSM issue 1092)
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testErrorInInitialAction() {

		// initial Action
		doThrow(ACTION_ERROR).when(actionMock).execute(any(StateContext.class));
		// OK -> container mechanism not needed to expose exception
		assertThatThrownBy(stateMachine::start).isEqualTo(ACTION_ERROR);
		// OK -> transition execution interrupted immediately
		assertThatStateMachineStateIsNull();
		// OK -> errorAction bypassed since errors should not be caught
		assertThatExtendedStateHoldsNoException();
	}

	/**
	 * When statemachine is started an exception occurs in its 1st transition S1 -> S2
	 */
	@Test
	// see above
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testExceptionInTransitionToS2() {

		// initial Action
		doNothing()
				// S1 -> S2
				.doThrow(ACTION_EXCEPTION).when(actionMock).execute(any(StateContext.class));
		// OK -> container mechanism not needed to expose exception
		assertThatThrownBy(stateMachine::start).isEqualTo(ACTION_EXCEPTION);
		assertThatExtendedStateHoldsExceptionsInOrder(ACTION_EXCEPTION);
		// NOK -> we cannot leave this state by an event which means we are caught
		assertThatStateMachineStateHasId(TestState.S1);
		// we also cannot leave it by stop + reset + start
		assertThatStateMachineResetDoesNotChangeState();

		// UnnecessaryStubbingException not available in case of unnecessary OnGoingStubbing, so we explicitly "verify"
		verify(actionMock, times(2)).execute(any(StateContext.class));
	}

	/**
	 * When statemachine is started an error occurs in its 1st transition S1 -> S2
	 */
	@Test
	// see above
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testErrorInTransitionToS2() {

		// initial Action
		doNothing()
				// S1 -> S2
				.doThrow(ACTION_ERROR).when(actionMock).execute(any(StateContext.class));
		// OK -> container mechanism not needed to expose exception
		assertThatThrownBy(stateMachine::start).isEqualTo(ACTION_ERROR);
		// OK -> errorAction bypassed
		assertThatExtendedStateHoldsNoException();
		// NOK -> we cannot leave this state by an event which means we are caught
		assertThatStateMachineStateHasId(TestState.S1);
		// we also cannot leave it by stop + reset + start
		assertThatStateMachineResetDoesNotChangeState();

		verify(actionMock, times(2)).execute(any(StateContext.class));
	}

	/**
	 * When statemachine is started an exception occurs in its S2 option guard 1.
	 */
	@ParameterizedTest(name = "{0}")
	@MethodSource("throwableArgumentsProvider")
	// see above
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testThrowableInS2OptionGuard(final String testLabel, final Throwable actionThrowable, final Throwable guardThrowable) {

		// S2 option 1 (first clause)
		doThrow(guardThrowable)
				// S2 option 2 (then clause)
				.doReturn(false)
				.when(guardMock).evaluate(any(StateContext.class));
		// NOK -> to access exception we need a container
		assertThatNoException().isThrownBy(stateMachine::start);
		assertThatExtendedStateHoldsExceptionsInOrder(guardThrowable);
		// NOK -> transition execution continues despite guard exception
		assertThatStateMachineStateHasId(TestState.S5);
		// state is terminal?
		assertThatStateMachineResetDoesNotChangeState();

		// initial action
		// S1 -> S2
		// Default S2 Option
		verify(actionMock, times(3)).execute(any(StateContext.class));
		verify(guardMock, times(2)).evaluate(any(StateContext.class));
	}

	/**
	 * When statemachine is started an exception occurs in S2 option action 1.
	 */
	@Test
	// see above
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testExceptionInS2OptionAction() {

		// S2 option 1
		doReturn(true).when(guardMock).evaluate(any(StateContext.class));
		// initial action
		doNothing()
				// S1 -> S2
				.doNothing()
				// S2 option 1 (first clause)
				.doThrow(ACTION_EXCEPTION)
				.when(actionMock).execute(any(StateContext.class));
		// OK -> container mechanism not needed to expose exception
		assertThatThrownBy(stateMachine::start).isEqualTo(ACTION_EXCEPTION);
		assertThatExtendedStateHoldsExceptionsInOrder(ACTION_EXCEPTION);
		// NOK -> we cannot leave this state by an event which means we are caught
		assertThatStateMachineStateHasId(TestState.S1);
		// we also cannot leave it by stop + reset + start
		assertThatStateMachineResetDoesNotChangeState();

		// OK -> transition execution interrupted immediately
		verify(actionMock, times(3)).execute(any(StateContext.class));
	}

	/**
	 * When statemachine is started an error occurs in S2 option action 1.
	 */
	@Test
	// see above
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testErrorInS2OptionAction() {

		// S2 option 1
		doReturn(true).when(guardMock).evaluate(any(StateContext.class));
		// initial action
		doNothing()
				// S1 -> S2
				.doNothing()
				// S2 option 1 (first clause)
				.doThrow(ACTION_ERROR)
				.when(actionMock).execute(any(StateContext.class));
		// OK -> container mechanism not needed to expose exception
		assertThatThrownBy(stateMachine::start).isEqualTo(ACTION_ERROR);
		// OK -> errorAction bypassed since errors should not be caught
		assertThatExtendedStateHoldsNoException();
		// NOK -> we cannot leave this state by an event which means we are caught
		assertThatStateMachineStateHasId(TestState.S1);
		// we also cannot leave it by stop + reset + start
		assertThatStateMachineResetDoesNotChangeState();

		// OK -> transition execution interrupted immediately
		verify(actionMock, times(3)).execute(any(StateContext.class));
	}

	/**
	 * When statemachine is started exceptions occur in all S2 options.
	 */
	@Test
	// see above
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testExceptionInAllS2Options() {

		// S2 option 1+2
		doThrow(GUARD_EXCEPTION).when(guardMock).evaluate(any(StateContext.class));
		// initial action
		doNothing()
				// S1 -> S2
				.doNothing()
				// default S2 option
				.doThrow(ACTION_EXCEPTION)
				.when(actionMock).execute(any(StateContext.class));
		// NOK -> this is collateral damage since we are not supposed to reach this point
		assertThatThrownBy(stateMachine::start).isEqualTo(ACTION_EXCEPTION);
		assertThatExtendedStateHoldsExceptionsInOrder(GUARD_EXCEPTION, GUARD_EXCEPTION, ACTION_EXCEPTION);
		// NOK -> we cannot leave this state by an event which means we are caught
		assertThatStateMachineStateHasId(TestState.S1);
		// we also cannot leave it by stop + reset + start
		assertThatStateMachineResetDoesNotChangeState();

		// NOK -> transition execution continues despite guard exception
		verify(actionMock, times(3)).execute(any(StateContext.class));
		verify(guardMock, times(2)).evaluate(any(StateContext.class));
	}

	/**
	 * When statemachine is started errors occur in all S2 options.
	 */
	@Test
	// see above
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testErrorsInAllS2Options() {

		// S2 option 1+2
		doThrow(GUARD_ERROR).when(guardMock).evaluate(any(StateContext.class));
		// initial action
		doNothing()
				// S1 -> S2
				.doNothing()
				// default S2 option
				.doThrow(ACTION_ERROR)
				.when(actionMock).execute(any(StateContext.class));
		// NOK -> this is collateral damage since we are not supposed to reach this point
		assertThatThrownBy(stateMachine::start).isEqualTo(ACTION_ERROR);
		assertThatExtendedStateHoldsExceptionsInOrder(GUARD_ERROR, GUARD_ERROR);
		// NOK -> we cannot leave this state by an event which means we are caught
		assertThatStateMachineStateHasId(TestState.S1);
		// we also cannot leave it by stop + reset + start
		assertThatStateMachineResetDoesNotChangeState();

		// NOK -> transition execution continues despite guard exception
		verify(actionMock, times(3)).execute(any(StateContext.class));
		verify(guardMock, times(2)).evaluate(any(StateContext.class));
	}

	/**
	 * When statemachine is started it reaches S3. The event is sent but an exception occurs
	 * in the transition action.
	 */
	@Test
	// see above
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testExceptionInS3Action() {

		// S2 option 1
		// S3 -> S4 - 1st + 2nd try
		doReturn(true).when(guardMock).evaluate(any(StateContext.class));
		// initial action
		doNothing()
				// S1 -> S2
				.doNothing()
				// S2 option 1
				.doNothing()
				// S3 -> S4
				.doThrow(ACTION_EXCEPTION)
				// S3 -> S4 on 2nd try
				// on entry to S4
				// in S4
				// on exit from S4
				.doNothing()
				.when(actionMock).execute(any(StateContext.class));

		assertThatNoException().isThrownBy(stateMachine::start);
		assertThatStateMachineStateHasId(TestState.S3);

		// NOK -> to access exception we need a container
		assertThatNoException().isThrownBy(() -> stateMachine.sendEvent(TestEvent.E));
		assertThatExtendedStateHoldsExceptionsInOrder(ACTION_EXCEPTION);
		// OK -> this is where we sent the event
		assertThatStateMachineStateHasId(TestState.S3);

		// machine reusable? try again!
		assertThatNoException().isThrownBy(() -> stateMachine.sendEvent(TestEvent.E));

		// NOK -> exit action executes late or twice
		verifyActionExecutedNumberOfTimesInRange(7, 9);
		verify(guardMock, times(3)).evaluate(any(StateContext.class));
	}

	/**
	 * When statemachine is started it reaches S3. The event is sent but an error occurs
	 * in the transition action.
	 */
	@Test
	// see above
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testErrorInS3Action() {

		// S2 option 1
		// S3 -> S4 - 1st + 2nd try
		doReturn(true).when(guardMock).evaluate(any(StateContext.class));
		// initial action
		doNothing()
				// S1 -> S2
				.doNothing()
				// S2 option 1
				.doNothing()
				// S3 -> S4
				.doThrow(ACTION_ERROR)
				// S3 -> S4 on 2nd try
				// on entry to S4
				// in S4
				// on exit from S4
				.doNothing()
				.when(actionMock).execute(any(StateContext.class));

		assertThatNoException().isThrownBy(stateMachine::start);
		assertThatStateMachineStateHasId(TestState.S3);

		// OK -> container mechanism not needed to expose exception
		assertThatThrownBy(() -> stateMachine.sendEvent(TestEvent.E)).isEqualTo(ACTION_ERROR);
		assertThatExtendedStateHoldsNoException();
		// OK -> this is where we sent the event
		assertThatStateMachineStateHasId(TestState.S3);

		// machine reusable? try again!
		assertThatNoException().isThrownBy(() -> stateMachine.sendEvent(TestEvent.E));

		// ? -> transition not taken on 2nd try
		verify(actionMock, times(4)).execute(any(StateContext.class));
		verify(guardMock, times(2)).evaluate(any(StateContext.class));
	}

	/**
	 * When statemachine is started it reaches S3. The event is sent but an exception occurs
	 * in the transition guard.
	 */
	@Test
	// see above
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testExceptionInS3Guard() {

		// S2 option 1
		doReturn(true)
				// S3 -> S4
				.doThrow(GUARD_EXCEPTION)
				// S3 -> S4 - 2nd try
				.doReturn(true)
				.when(guardMock).evaluate(any(StateContext.class));

		assertThatNoException().isThrownBy(stateMachine::start);
		assertThatStateMachineStateHasId(TestState.S3);

		// NOK -> to access exception we need a container
		assertThatNoException().isThrownBy(() -> stateMachine.sendEvent(TestEvent.E));
		assertThatExtendedStateHoldsExceptionsInOrder(GUARD_EXCEPTION);
		// OK -> this is where we sent the event
		assertThatStateMachineStateHasId(TestState.S3);

		// machine reusable? try again!
		assertThatNoException().isThrownBy(() -> stateMachine.sendEvent(TestEvent.E));

		verify(guardMock, times(3)).evaluate(any(StateContext.class));
		// initial action
		// S1 -> S2
		// S2 option 1
		// S3 -> S4 on 2nd try
		// on entry to S4
		// in S4
		// on exit from S4

		// NOK -> exit action executes late or twice
		verifyActionExecutedNumberOfTimesInRange(6, 8);
	}

	/**
	 * When statemachine is started it reaches S3. The event is sent but an error occurs
	 * in the transition guard.
	 */
	@Test
	// see above
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testErrorInS3Guard() {

		// S2 option 1
		doReturn(true)
				// S3 -> S4
				.doThrow(GUARD_ERROR)
				// S3 -> S4 - 2nd try
				.doReturn(true)
				.when(guardMock).evaluate(any(StateContext.class));

		assertThatNoException().isThrownBy(stateMachine::start);
		assertThatStateMachineStateHasId(TestState.S3);

		// OK -> container mechanism not needed to expose exception
		assertThatThrownBy(() -> stateMachine.sendEvent(TestEvent.E)).isEqualTo(GUARD_ERROR);
		assertThatExtendedStateHoldsExceptionsInOrder(GUARD_ERROR);
		// OK -> this is where we sent the event
		assertThatStateMachineStateHasId(TestState.S3);

		// machine reusable? try again!
		assertThatNoException().isThrownBy(() -> stateMachine.sendEvent(TestEvent.E));

		// ? -> transition not taken on 2nd try
		verify(guardMock, times(2)).evaluate(any(StateContext.class));
		// initial action
		// S1 -> S2
		// S2 option 1
		verify(actionMock, times(3)).execute(any(StateContext.class));
	}

	/**
	 * When statemachine is started it reaches S4. An exception occurs in its entry,
	 * behavior and exit action.
	 */
	@Test
	// see above
	@SuppressWarnings("unchecked")
	void testExceptionInAllS4Actions() {

		// S2 option 1
		doReturn(false)
				// S2 option 2
				.doReturn(true)
				.when(guardMock).evaluate(any(StateContext.class));
		// initial action
		doNothing()
				// S1 -> S2
				.doNothing()
				// S2 option 2
				.doNothing()
				// on entry to S4
				// in S4
				// on exit from S4
				.doThrow(ACTION_EXCEPTION)
				.when(actionMock).execute(any(StateContext.class));

		// NOK -> to access exception we need a container
		// NOK -> appears to be a bug - at least this is not what we would expect as an exception
		assertThatThrowableIfThrownByStartStateMachineIsOfAnyClassIn(ConcurrentModificationException.class);
		// NOK -> exit action appears to be executed late or twice sometimes
		assertThatExtendedStateHoldsActionExceptionTimesBetween(2, 4);
		// NOK -> transition execution continues despite action exception
		// NOK -> according to logs we always reach S5 but the current state is sometimes reported to be
		//        S4 (...a bug when setting AbstractStateMachine#lastState ?)
		assertThatStateMachineStateHasOneOfIds(TestState.S4, TestState.S5);
		// we also cannot leave state by stop + reset + start
		assertThatStateMachineResetDoesNotChangeState();

		verify(guardMock, times(2)).evaluate(any(StateContext.class));
		verifyActionExecutedNumberOfTimesInRange(5, 7);
	}

	/**
	 * When statemachine is started it reaches S4. An error occurs in its entry action.
	 */
	@Test
	// see above
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testErrorInS4EntryAction() {

		// S2 option 1
		doReturn(false)
				// S2 option 2
				.doReturn(true)
				.when(guardMock).evaluate(any(StateContext.class));
		// initial action
		doNothing()
				// S1 -> S2
				.doNothing()
				// S2 option 2
				.doNothing()
				// on entry to S4
				.doThrow(ACTION_ERROR)
				.when(actionMock).execute(any(StateContext.class));

		// OK -> container mechanism not needed to expose exception
		assertThatThrownBy(stateMachine::start).isEqualTo(ACTION_ERROR);
		assertThatExtendedStateHoldsNoException();
		// OK -> transition execution interrupted immediately
		assertThatStateMachineStateHasId(TestState.S4);
		// cannot leave state by stop + reset + start
		assertThatStateMachineResetDoesNotChangeState();

		verify(guardMock, times(2)).evaluate(any(StateContext.class));
		verify(actionMock, times(4)).execute(any(StateContext.class));
	}

	/**
	 * When statemachine is started it reaches S4. An error occurs in its behavior action.
	 */
	@Test
	// see above
	@SuppressWarnings({ "unchecked", "deprecation" })
	void testErrorInS4BehaviorAction() {

		// S2 option 1
		doReturn(false)
				// S2 option 2
				.doReturn(true)
				.when(guardMock).evaluate(any(StateContext.class));
		// initial action
		doNothing()
				// S1 -> S2
				.doNothing()
				// S2 option 2
				.doNothing()
				// on entry to S4
				.doNothing()
				// in S4
				.doThrow(ACTION_ERROR)
				// on exit from S4
				.doNothing()
				.when(actionMock).execute(any(StateContext.class));

		// NOK -> to access exception we need a container
		assertThatNoException().isThrownBy(stateMachine::start);
		// NOK -> since error action is not executed in case of error (...but only for exceptions...)
		// we do not know about the error
		assertThatExtendedStateHoldsNoException();
		// NOK -> transition execution continues despite behavior action exception
		assertThatStateMachineStateHasId(TestState.S5);
		// state is terminal?
		assertThatStateMachineResetDoesNotChangeState();

		verify(guardMock, times(2)).evaluate(any(StateContext.class));
		verify(actionMock, times(6)).execute(any(StateContext.class));
	}

	/**
	 * When statemachine is started it reaches S4. An error occurs in its exit action.
	 */
	@Test
	// see above
	@SuppressWarnings("unchecked")
	void testErrorInS4ExitAction() {

		// S2 option 1
		doReturn(false)
				// S2 option 2
				.doReturn(true)
				.when(guardMock).evaluate(any(StateContext.class));
		// initial action
		doNothing()
				// S1 -> S2
				.doNothing()
				// S2 option 2
				.doNothing()
				// on entry to S4
				.doNothing()
				// in S4
				.doNothing()
				// on exit from S4
				.doThrow(ACTION_ERROR)
				.when(actionMock).execute(any(StateContext.class));

		// NOK -> appears to be a bug - behavior is inconsistent
		assertThatThrowableIfThrownByStartStateMachineIsOfAnyClassIn(ConcurrentModificationException.class, ACTION_ERROR.getClass());
		assertThatExtendedStateHoldsNoException();
		// NOK -> ...a bug when setting AbstractStateMachine#lastState ?
		assertThatStateMachineStateHasOneOfIds(TestState.S4, TestState.S5);
		// we cannot leave state by stop + reset + start
		assertThatStateMachineResetDoesNotChangeState();

		verify(guardMock, times(2)).evaluate(any(StateContext.class));
		// NOK -> exit action appears to be executed late or twice sometimes
		verifyActionExecutedNumberOfTimesInRange(5, 7);
	}

	private void assertThatExtendedStateHoldsExceptionsInOrder(final Throwable... exceptions) {

		assertThat(stateMachine)
				.extracting(StateMachine::getExtendedState)
				.extracting(ExtendedStateExceptionAccessor::getExceptions, as(InstanceOfAssertFactories.LIST))
				.containsExactlyElementsOf(List.of(exceptions));

	}

	private void assertThatExtendedStateHoldsNoException() {
		assertThatExtendedStateHoldsExceptionsInOrder();
	}

	// read-only varargs access
	@SafeVarargs
	// see above
	@SuppressWarnings("deprecation")
	private void assertThatThrowableIfThrownByStartStateMachineIsOfAnyClassIn(final Class<? extends Throwable>... exceptionClasses) {
		try {
			stateMachine.start();
		} catch (final Throwable e) {
			assertThat(e).isOfAnyClassIn(exceptionClasses);
		}
	}

	private void assertThatExtendedStateHoldsActionExceptionTimesBetween(final int atLeastTimes, final int atMostTimes) {

		assertThat(stateMachine)
				.extracting(StateMachine::getExtendedState)
				.extracting(ExtendedStateExceptionAccessor::getExceptions, as(InstanceOfAssertFactories.LIST))
				.hasSizeBetween(atLeastTimes, atMostTimes)
				.containsOnly(ACTION_EXCEPTION);
	}

	// see above
	@SuppressWarnings("unchecked")
	private void verifyActionExecutedNumberOfTimesInRange(final int atLeastTimes, final int atMostTimes) {
		verify(actionMock, atLeast(atLeastTimes)).execute(any(StateContext.class));
		verify(actionMock, atMost(atMostTimes)).execute(any(StateContext.class));
	}

	private void assertThatStateMachineStateIsNull() {

		assertThat(stateMachine)
				.extracting(StateMachine::getState)
				.isNull();
	}

	private void assertThatStateMachineStateHasOneOfIds(final TestState... stateIds) {

		assertThat(stateMachine)
				.extracting(StateMachine::getState)
				.isNotNull()
				.extracting(State::getId)
				.isIn(List.of(stateIds));
	}

	private void assertThatStateMachineStateHasId(@Nullable final TestState stateId) {

		assertThat(stateMachine.getState() == null ? null : stateMachine.getState().getId())
				.isEqualTo(stateId);
	}

	// s.o.
	@SuppressWarnings("deprecation")
	private void assertThatStateMachineResetDoesNotChangeState() {

		final State<TestState, TestEvent> state = stateMachine.getState();
		final TestState currentStateId = state == null ? null : state.getId();

		stateMachine.stop();
		final StateMachineContext<TestState, TestEvent> context = new DefaultStateMachineContext<>(currentStateId, null, Map.of(),
				stateMachine.getExtendedState());
		stateMachine.getStateMachineAccessor().doWithAllRegions(access -> access.resetStateMachine(context));

		assertThatNoException().isThrownBy(stateMachine::start);
		assertThatStateMachineStateHasId(currentStateId);
	}

	private static Stream<Arguments> throwableArgumentsProvider() {
		return Stream.of(
				Arguments.of(EXCEPTION_TEST_LABEL, ACTION_EXCEPTION, GUARD_EXCEPTION),
				Arguments.of(ERROR_TEST_LABEL, ACTION_ERROR, GUARD_ERROR));
	}

	@EnableStateMachineFactory
	@AllArgsConstructor(access = AccessLevel.PACKAGE)
	static class TestStateMachineConfigurer extends StateMachineConfigurerAdapter<TestState, TestEvent> {

		private final Action<TestState, TestEvent> actionMock;

		private final Guard<TestState, TestEvent> guardMock;

		@Override
		public void configure(final StateMachineStateConfigurer<TestState, TestEvent> states) throws Exception {

			states.withStates()

					.initial(TestState.S1, actionWithExH())
					.choice(TestState.S2)
					.state(TestState.S3)
					.state(TestState.S4, actionWithExH(), actionWithExH())
					.stateDo(TestState.S4, actionWithExH())
					.end(TestState.S5);
		}

		private Action<TestState, TestEvent> actionWithExH() {
			return Actions.errorCallingAction(actionMock, ExceptionHandlingAction::execute);
		}

		private Guard<TestState, TestEvent> guardWithExH() {
			return ExceptionHandlingGuardWrapper.wrap(guardMock);
		}

		@Override
		public void configure(final StateMachineTransitionConfigurer<TestState, TestEvent> transitions) throws Exception {

			transitions

					.withExternal()
					.source(TestState.S1)
					.target(TestState.S2)
					.action(actionWithExH())
					.and()

					.withChoice()
					.source(TestState.S2)
					.first(TestState.S3, guardWithExH(), actionWithExH())
					.then(TestState.S4, guardWithExH(), actionWithExH())
					.last(TestState.S5, actionWithExH())
					.and()

					.withExternal()
					.source(TestState.S3)
					.target(TestState.S4)
					.event(TestEvent.E)
					.guard(guardWithExH())
					.action(actionWithExH())
					.and()

					.withExternal()
					.source(TestState.S4)
					.target(TestState.S5);
		}
	}

	static class BeanConfig {

		@Bean
		// a simple mock, i.e. generics can be ignored
		@SuppressWarnings("unchecked")
		Action<TestState, TestEvent> action() {
			return mock(Action.class);
		}

		@Bean
		// see above
		@SuppressWarnings("unchecked")
		Guard<TestState, TestEvent> guard() {
			return mock(Guard.class);
		}
	}

	private enum TestState {

		S1,
		S2,
		S3,
		S4,
		S5
	}

	private enum TestEvent {

		E
	}
}